//
//  FJDPersistentWindowController.m
//  FJDCoreDataWindow
//
//  Created by FUJIDANA on 06/08/17.
//  Copyright 2006 FUJIDANA. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#import "FJDPersistentWindowController.h"
#import "FJDCoreDataAppDelegate.h"


@implementation FJDPersistentWindowController

#pragma mark Class methods

+ (NSURL *)URLWithPersistentStoreDirectory:(NSString *)directory filename:(NSString *)filename
{
	NSArray *paths = NSSearchPathForDirectoriesInDomains(NSApplicationSupportDirectory, NSUserDomainMask, YES);
	NSString *basePath = ([paths count] > 0) ? [paths objectAtIndex:0] : NSTemporaryDirectory();
	NSString *pStoreDirectoryPath = [basePath stringByAppendingPathComponent:directory];
	
	NSFileManager *fileManager = [NSFileManager defaultManager];
	
	if (![fileManager fileExistsAtPath:pStoreDirectoryPath isDirectory:NULL]) {
		[fileManager createDirectoryAtPath:pStoreDirectoryPath attributes:nil];
	}
	return [NSURL fileURLWithPath:[pStoreDirectoryPath stringByAppendingPathComponent:filename]];
}

#pragma mark Initializing methods

- (id)initWithWindow:(NSWindow *)window URLForPersistentStore:(NSURL *)url storeType:(NSString *)storeType
{
	self = [super initWithWindow:window];
	if (self != nil) {
		_pStoreURL  = [url copy];
		_pStoreType = [storeType copy];
	}
	return self;
}

- (id)initWithWindowNibName:(NSString *)windowNibName URLForPersistentStore:(NSURL *)url storeType:(NSString *)storeType
{
	self = [super initWithWindowNibName:windowNibName];
	if (self != nil) {
		_pStoreURL  = [url copy];
		_pStoreType = [storeType copy];
	}
	return self;
}

- (id)initWithWindowNibName:(NSString *)windowNibName owner:(id)owner URLForPersistentStore:(NSURL *)url storeType:(NSString *)storeType
{
	self = [super initWithWindowNibName:windowNibName owner:owner];
	if (self != nil) {
		_pStoreURL  = [url copy];
		_pStoreType = [storeType copy];
	}
	return self;
}

- (id)initWithWindowNibPath:(NSString *)windowNibPath owner:(id)owner URLForPersistentStore:(NSURL *)url storeType:(NSString *)storeType
{
	self = [super initWithWindowNibPath:windowNibPath owner:owner];
	if (self != nil) {
		_pStoreURL  = [url copy];
		_pStoreType = [storeType copy];
	}
	return self;
}

- (void)dealloc
{
    [_managedObjectContext release], _managedObjectContext = nil;
    [_persistentStoreCoordinator release], _persistentStoreCoordinator = nil;
    [_managedObjectModel release], _managedObjectModel = nil;
	[_pStoreURL release], _pStoreURL = nil;
	[_pStoreType release], _pStoreType = nil;
    [super dealloc];
}

#pragma mark Accessor methods

- (NSManagedObjectModel *)managedObjectModel
{
	if (_managedObjectModel != nil) {
		return _managedObjectModel;
	}
	
	NSMutableSet *allBundles = [[NSMutableSet alloc] init];
	[allBundles addObject: [NSBundle mainBundle]];
	[allBundles addObjectsFromArray:[NSBundle allFrameworks]];
	
	_managedObjectModel = [[NSManagedObjectModel mergedModelFromBundles:[allBundles allObjects]] retain];
	[allBundles release];
	
	return _managedObjectModel;
}

- (NSPersistentStoreCoordinator *)persistentStoreCoordinator
{
	if (_persistentStoreCoordinator != nil) {
		return _persistentStoreCoordinator;
	}
	
	NSError *error;
	_persistentStoreCoordinator = [[NSPersistentStoreCoordinator alloc] initWithManagedObjectModel:[self managedObjectModel]];
	
	if (![_persistentStoreCoordinator addPersistentStoreWithType:_pStoreType
												   configuration:nil
															 URL:_pStoreURL
														 options:nil
														   error:&error]) {
		[[NSApplication sharedApplication] presentError:error];
	}
	
	[[FJDCoreDataAppDelegate applicationDelegate] addPersistentWindowController:self];
	
	return _persistentStoreCoordinator;
}

- (NSManagedObjectContext *)managedObjectContext
{
	if (_managedObjectContext != nil) {
		return _managedObjectContext;
	}
	
	NSPersistentStoreCoordinator *coordinator = [self persistentStoreCoordinator];
	if (coordinator != nil) {
		_managedObjectContext = [[NSManagedObjectContext alloc] init];
		[_managedObjectContext setPersistentStoreCoordinator:coordinator];
	}
	
	return _managedObjectContext;
}


#pragma mark Mehtods delegated by NSWindow

- (NSUndoManager *)windowWillReturnUndoManager:(NSWindow *)window
{
	return [[self managedObjectContext] undoManager];
}

#pragma mark Action methods

- (IBAction)saveAction:(id)sender
{
	NSError *error = nil;
	if (![[self managedObjectContext] save:&error]) {
		[self presentError:error];
	}
}

- (IBAction)revertAction:(id)sender
{
	NSAlert *alert = [[[NSAlert alloc] init] autorelease];
	[alert addButtonWithTitle:NSLocalizedStringFromTable(@"Revert", @"CoreDataLocalizable", @"revertSheet.firstButton")];
	[alert addButtonWithTitle:NSLocalizedStringFromTable(@"Cancel", @"CoreDataLocalizable", @"revertSheet.secondButton")];
	[alert setMessageText:NSLocalizedStringFromTable(@"Do you want to revert to the most recently saved revision?", @"CoreDataLocalizable", @"revertSheet.message")];
	[alert setInformativeText:NSLocalizedStringFromTable(@"Reverting will cause your current database to be overwirtten. This operation can't be undone.", @"CoreDataLocalizable", @"revertSheet.information")];	
	[alert setAlertStyle:NSInformationalAlertStyle];
	
	[alert beginSheetModalForWindow:[self window]
					  modalDelegate:self
					 didEndSelector:@selector(revertActionAlertDidEnd:returnCode:contextInfo:)
						contextInfo:NULL];
}

- (void)revertActionAlertDidEnd:(NSAlert *)alert returnCode:(int)returnCode contextInfo:(void *)contextInfo
{
	if (returnCode == NSAlertFirstButtonReturn) {
		[[self managedObjectContext] processPendingChanges];
		[[self managedObjectContext] rollback];
		//		[self didManagedObjectContextRollback];
	}
}

#pragma mark Methods overriding NSResponder

// Based on "NSPersistentDocument Core Data Tutorial
// > Localizing and Customizing Model Property Names and Error Messages"
// > Customizing the Document Alert Panel", 
// But I modified several part so that unmodified errors should be passed to superclass.
// (See "Handling Received Errors" section in "Error Handling Programming Guide For Cocoa".)

- (NSError *)willPresentError:(NSError *)inError
{
	if ([[inError domain] isEqualToString:NSCocoaErrorDomain] 
		&& [inError code] == NSValidationMultipleErrorsError) {
		
		NSArray *detailedErrors = [[inError userInfo] objectForKey:NSDetailedErrorsKey];
		
		// present error messages for up to 3 validation errors at a time.
		unsigned numErrors = [detailedErrors count];
		NSMutableString *errorString = [NSMutableString stringWithFormat:NSLocalizedStringFromTable(@"%u validation errors have occurred", @"CoreDataLocalizable", nil), numErrors];
		if (numErrors > 3) {
			[errorString appendFormat:NSLocalizedStringFromTable(@".\nThe first 3 are:\n", @"CoreDataLocalizable", nil)];
		} else {
			[errorString appendFormat:@":\n"];
		}
		unsigned i, displayErrors = numErrors > 3 ? 3 : numErrors;
		for (i = 0; i < displayErrors; i++) {
			[errorString appendFormat:@"%@\n",
				[[detailedErrors objectAtIndex:i] localizedDescription]];
		}
		
		// Create a new error with the new userInfo
		NSMutableDictionary *newUserInfo = [NSMutableDictionary
					dictionaryWithDictionary:[inError userInfo]];
		[newUserInfo setObject:errorString forKey:NSLocalizedDescriptionKey];
		
		return [NSError errorWithDomain:[inError domain] code:[inError code] userInfo:newUserInfo];  
	} else {
		return [super willPresentError:inError];
	}
}


#pragma mark Methods implementing NSToolbarItemValidation informal protocol

- (BOOL)validateToolbarItem:(NSToolbarItem *)theItem
{
	if ([theItem action] == @selector(saveAction:)) {
		return [[self managedObjectContext] hasChanges];
	} else if ([theItem action] == @selector(revertAction:)) {
		return [[self managedObjectContext] hasChanges];
	}
	return YES;
}


#pragma mark Methods implementing NSMenuValidation informal protocol

- (BOOL)validateMenuItem:(id <NSMenuItem>)theItem
{
	if ([theItem action] == @selector(saveAction:)) {
		return [[self managedObjectContext] hasChanges];
	} else if ([theItem action] == @selector(revertAction:)) {
		return [[self managedObjectContext] hasChanges];
	}
	return YES;
}

@end
